/* ///////////////////////////////////////////////////////////////////////// */
/*  This is part of the source of the OMAP 5912 heterogeneous dual-core      */
/*  MPEG-4 SP video decoder published in ACM Transactions on Embedded        */
/*  Computing Systems, Vol. X, Issue Y.                                      */
/* ------------------------------------------------------------------------- */
/*  The source code is released under GPL license.                           */
/*                                                                           */
/*  Copyright, 2011                                                          */
/*  Multimedia Embedded Systems Labs                                         */
/*  Dept. of Computer Science                                                */
/*  National Chiao Tung University                                           */
/*  Hsinchu, Taiwan.                                                         */
/* ///////////////////////////////////////////////////////////////////////// */

/** ============================================================================
 *   @file  icache_example.c
 *
 *   @path  $(CSLPATH)\dsp\icache\example
 *
 *   @desc  Example function to test DSP I-cache CSL implementation.
 *
 * =============================================================================
 *   @n<b> Example Description </b>
 *   @n This example shows how to configure the I-cache in 1-Way and 1-Half 
 *      ramset configuration. For verifying that I-cache has been enabled the
 *      function verifies that the I-cache Enabled bit is ISR register is set
 *      once the cache is enabled. For verifying the ramset fill operation
 *      the function also verifies that the Tag Valid bit in the RCR1 register
 *      is set after configuring the TAG in RTR1 register. Before running this
 *      example it is necessary to configure DSP MMU so as to enable access to
 *      extenal memory interface (SDRAM). Code for this example is loaded into
 *      the SDRAM and is executed from there. For details regarding configuring
 *      DSP MMU please refer to the readme file contained in the example folder
 *
 *      The main example function is:
 *
 *        (a) <b> icacheConfigurationExample </b>
 *             The function configures the I-cache
 *
 *   <b> Test Procedure </b>
 *   @verbatim
         1. Connect the 5912 board to the XDS560 emulator on
            the host system
         2. Power on the board
         3. Configure the CCS setup for OMAP1610XDS560 and configure 
            the gel file tms470_helen2.gel for ARM & 
            c5000_helen2_sk.gel for DSP.
         4. From the parallel debug manager, launch CCS window
            corresponding to DSP
         5. From the parallel debug manager, launch CCS window
            corresponding to ARM
         6. On the ARM side CCS window, open the dspIcache_proj_arm
            project. Provide path for the prompted files and libraries
         7. On the DSP side CCS window, open the dspicache_example_dsp
            project.   
         7. On the ARM side CCS window, load the program 
           "dspicache_proj_arm.out" and execute it.
         8. On the DSP side CCS window, load the program 
            "dspicache_example_dsp.out" and execute it.
                        
     @endverbatim
 *
 *   <b> Test Observation </b>
 *   @verbatim
         Result of the example will be logged on the stdout of CCS DSP window
     @endverbatim                                               
 *   
 * ===========================================================================
 */

/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *  ============================================================================
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  07-May-2004 sks File Created.
 *  16-Jul-2004 rr  Modified according to the new CSL architecture
 *
 * =============================================================================
 */

#include <soc.h>
#include <csl_icache.h>
#include <stdio.h>
//#include <csl_test_fw.h>

/** Object structure for DSP I-cache CSL */
CSL_IcacheObj icacheObj;

//#define ICACHE_RAMSET_TAG       (0x0000C04)
#define ICACHE_RAMSET_TAG       (0x0000080)
#define DELAY_COUNT             (0xFFFF)

/* Local function prototypes */

void    icacheConfigurationExample(void);

CSL_IcacheHandle icacheInitCSL(CSL_Status * status);

void    icacheDelayFunc(void);

/** ============================================================================
 *   @n@b main
 *
 *   @b Description
 *   @n Main function to invloke example function
 *      
 *   @b Arguments
 *   @verbatim
            None
     @endverbatim
 *
 *   <b> Return Value </b>  void
 *   @n                     None
 *
 *   @b Example
 *   @verbatim
           main ();
     @endverbatim
 * ===========================================================================
 */
CSL_IcacheHandle hIcache;
CSL_Status status;
void
i_cache_open(void)
{
    //printf ("\n************* Running I-cache Configuration Example ***********\n");
    icacheConfigurationExample();
    //printf ("\n*************** End I-cache Configuration Example ***********\n");
}

void
i_cache_close(void)
{
    if (status != CSL_SOK)
    {
        //puts("9. Cache disable command failed\n");
        //printf("I-cache error in 9.\n");
        //PRINT_FAIL("icache", "icache_example", "CSL_icacheHwControl3");
        CSL_icacheClose(hIcache);
        return;
    }

    printf("I-cache has been disabled successfully\n");

    /* Close the I-cache instance */
    status = CSL_icacheClose(hIcache);
    if (status != CSL_SOK)
    {
        //puts("10. I-cache close function failed\n");
        //printf("I-cache error in 10.\n"); 
        //PRINT_FAIL("icache", "icache_example", "CSL_icacheClose");
    }
}

/** ============================================================================
 *   @n@b icacheConfigurationExample
 *
 *   @b Description
 *   @n This function shows how to configure the I-cache in 
 *      1-Way and 1-Half ramset organization. The function 
 *      algorithm is as follows:
 *
 *   @li Initialize I-cache CSL
 *
 *   @li Open I-cache instance
 *
 *   @li Configure I-cache in 1-Way and 1-Half ramset 
 *       organization
 *
 *   @li Enable the configured I-cache organization
 *
 *   @li Verify that I-cache has been enabled. This is 
 *       verified by querying the I-cache Enabled field
 *       in the ISR register.
 *
 *   @li Set the Tag field in the RTR1 register to cache
 *       a block of 4-Kbytes
 *
 *   @li Verify that the ramset fill is successful. This
 *       is verified by querying the Tag Valid field in the
 *       RCR1 register. It should be 1 (set).
 *
 *   @li Disable the I-cache
 *
 *   @li Close the I-cache instance
 *
 *   @b Arguments
 *   @verbatim
            hwSetup        Hardware setup for configuring hardware
                                        
            status         Filed in which to return status of the function
     @endverbatim
 *
 *   <b> Return Value </b>  void
 *   @n                     None
 *
 *   @b Example
 *   @verbatim
           icacheConfigurationExample ();
     @endverbatim
 * ===========================================================================
 */

void
icacheConfigurationExample(void)
{
    // CSL_Status                      status = CSL_SOK;
    //CSL_IcacheHandle                hIcache;

    CSL_IcacheEnableDisable enblDisblStatus;
    CSL_IcacheHfRamsetCfg ramsetCfg;
    CSL_IcacheHwSetup hwSetup;
    status = CSL_SOK;
    /* 
     * Initialize and open the DSP I-cache Instance without hardware 
     * setup 
     */
    hIcache = icacheInitCSL(&status);

    if (status != CSL_SOK)
    {
        //printf("I-cache error in 1.\n");
        //puts("1. Error opening the instance\n");
        //PRINT_FAIL("icache", "icache_example", "icacheInitCSL");
        return;
    }

    hwSetup.disableClk = CSL_ICACHE_CFG_PARAM_ENABLE;
    hwSetup.autoGating = CSL_ICACHE_CFG_PARAM_DISABLE;
    /* Select cache organization as 1-Way and 1 half ramset */
    //hwSetup.icacheOrgn = CSL_ICACHE_1WAY_1HFRAMSET_ORG;
    hwSetup.icacheOrgn = CSL_ICACHE_2WAY_2HFRAMSET_ORG;
    hwSetup.icacheGlobalFlush = CSL_ICACHE_GLOBAL_FLUSH_DISABLE;
    hwSetup.icacheGlobalEnable = CSL_ICACHE_GLOBAL_ENABLE_ENABLE;

    /* Call hardware setup function to setup  */
    status = CSL_icacheHwSetup(hIcache, &hwSetup);

    if (status != CSL_SOK)
    {
        //printf("I-cache error in 2.\n");
        //puts("2. Cache configuration failed\n");
        //PRINT_FAIL("icache", "icache_example", "CSL_icacheHwSetup");

        CSL_icacheClose(hIcache);
        return;
    }

    /* Enable the I-cache */
    status = CSL_icacheHwControl(hIcache, CSL_ICACHE_CMD_CACHE_ENABLE, NULL);
    if (status != CSL_SOK)
    {
        //printf("I-cache error in 3.\n");
        //puts("3. Cache Enable command failed\n");
        //PRINT_FAIL("icache", "icache_example", "CSL_icacheHwControl1");
        CSL_icacheClose(hIcache);
        return;
    }

    /* 
     * Query the I-cache Enabled field in ISR register, so as to confirm
     * that cache has been enabled
     */
    status = CSL_icacheGetHwStatus(hIcache,
                                   CSL_ICACHE_QUERY_IS_ENABLED,
                                   (void *) &enblDisblStatus);

    if (status != CSL_SOK)
    {
        //printf("I-cache error in 4.\n");
        //puts("4. CSL_icacheGetHwStatus Failed\n");
        //PRINT_FAIL("icache", "icache_example", "CSL_icacheGetHwStatus1");

        CSL_icacheClose(hIcache);
        return;
    }
    else
    {
        if (enblDisblStatus != CSL_ICACHE_CFG_PARAM_ENABLE)
        {
            //printf("I-cache error in 5.\n");
            //puts ("5. Icache is not enabled\n"); 
            //PRINT_FAIL("icache", "icache_example", "I-Cache is not enabled");
            CSL_icacheClose(hIcache);
            return;
        }
    }

    /* Initialize Tag field configuration */
    ramsetCfg.hfRamsetNumber = CSL_ICACHE_1HFRAMSET;
    ramsetCfg.hfRamsetTag = (Uint16) ICACHE_RAMSET_TAG;

    /* Set Tag in the RTR1 */
    status =
        CSL_icacheHwControl(hIcache, CSL_ICACHE_CMD_SET_HFRAMSET_TAG,
                            (void *) &ramsetCfg);

    if (status != CSL_SOK)
    {
        //printf("I-cache error in 6.\n");
        //puts ("6. Tag field setting failed\n");
        //PRINT_FAIL("icache", "icache_example", "CSL_icacheHwControl2");
        CSL_icacheClose(hIcache);
        return;
    }

    /* Ramset load operation can take some time so wait for some time */
    // icacheDelayFunc ();

    /* Verify that ramset fill has completed (Query Tag Valid field in RTR1) */
    status =
        CSL_icacheGetHwStatus(hIcache, CSL_ICACHE_QUERY_HFRAMSET_CFG,
                              (void *) &ramsetCfg);

    if (status != CSL_SOK)
    {
        //printf("I-cache error in 7.\n");
        //puts("7. CSL_icacheGetHwStatus Failed\n");
        //PRINT_FAIL("icache", "icache_example", "CSL_icacheGetHwStatus2");
        CSL_icacheClose(hIcache);
        return;
    }
    else
    {
        if (ramsetCfg.hfRamsetTagValid != TRUE)
        {
            //printf("I-cache error in 8.\n");
            //puts ("8. Tag Valid field is not set\n"); 
            //PRINT_FAIL("icache", "icache_example", "Tag Valid field is not set");
            CSL_icacheClose(hIcache);
            return;
        }
    }
    printf("I-cache has been initialized and enabled successfully\n");
    //puts ("I-cache has been initialized and enabled successfully\n");

    /* Disable the cache */
    // status = CSL_icacheHwControl (hIcache , CSL_ICACHE_CMD_CACHE_DISABLE, NULL);

    //puts ("I-cache configuration example is successful\n");
    //PRINT_PASS("icache", "icache_example", "icache_example");

    return;

}

/** ============================================================================
 *   @n@b icacheInitCSL
 *
 *   @b Description
 *   @n This is a utility function used by the main example function for 
 *      initializing and opening the DSP I-cache module
 *      
 *   @b Arguments
 *   @verbatim
             status         Filed in which to return status of the function
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_IcacheHandle
 *   @n                     Valid handle to opened instance if successful
 *
 *   @b Example
 *   @verbatim
           icacheInitCSL ( &status);
     @endverbatim
 * ===========================================================================
 */

CSL_IcacheHandle
icacheInitCSL(CSL_Status * status)
{

    CSL_IcacheHandle hIcacheLcl = NULL;

    /* Initialize the DSP I-cache CSL */
    CSL_icacheInit(NULL);

    /* Open the DSP I-cache Instance */
    hIcacheLcl = CSL_icacheOpen(&icacheObj, CSL_ICACHE, NULL, status);

    return (hIcacheLcl);
}

/** ============================================================================
 *   @n@b icacheDelayFunc
 *
 *   @b Description
 *   @n This is a utility function which has been used for introducing some
 *      delay before executing further steps in the main example function
 *      
 *   @b Arguments
 *   @verbatim
            None
     @endverbatim
 *
 *   <b> Return Value </b>  void
 *   @n                     None
 *
 *   @b Example
 *   @verbatim
           icacheDelayFunc ();
     @endverbatim
 * ===========================================================================
 */

//void icacheDelayFunc (void)
//{
//    Uint32  count = 0;
//
//    while (count < DELAY_COUNT)
//        count++;
//}
